/* --------------------------------------------------------------
  Language.js 2019-06-07
  Gambio GmbH
  http://www.gambio.de
  Copyright (c) 2019 Gambio GmbH
  Released under the GNU General Public License (Version 2)
  [http://www.gnu.org/licenses/gpl-2.0.html]
  --------------------------------------------------------------*/


'use strict';

/**
 * Language Library
 */
export default class Language {
	/**
	 * Language Class
	 *
	 * This class handles the localization for the JavaScript code of StyleEdit. Once there
	 * are sections added you can use the "translate" method for fetching the translated string.
	 */
	constructor() {
		/**
		 * Translation Sections
		 *
		 * @type {Object}
		 */
		this.sections = {};
	}
	
	/**
	 * Add a new translation section.
	 *
	 * @param {String} name The name of the section to be used when fetching a translation later.
	 * @param {Object} translations A key-translation pairs object that will be used later for
	 * getting the correct strings (see translate method).
	 *
	 * @return {Language} Returns the object instance.
	 */
	addSection(name, translations) {
		StyleEdit.Validator.isString(name);
		StyleEdit.Validator.isObject(translations);
		
		if (this.sections[name] !== undefined) {
			StyleEdit.Debug.warn(`The provided section name already exists and will be overridden: "${name}"`);
		}
		
		this.sections[name] = translations;
		
		return this;
	};
	
	/**
	 * Translate the provided key of the provided section.
	 *
	 * @param {String} key The key of the translation.
	 * @param {String} section The section name the key belongs to.
	 *
	 * @return {String} Returns the translation.
	 *
	 * @throws Error If the translation was not found.
	 */
	translate(key, section) {
		StyleEdit.Validator.isString(key);
		StyleEdit.Validator.isString(section);
		
		if (this.sections[section] === undefined) {
			throw new Error(`Requested section was not found: "${section}"`);
		}
		
		return this.sections[section][key] || '{' + key + '}';
	};
}
